<?php
// Turn on all error reporting
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

echo "<h3>Simple Payroll Creation Debug</h3>";

// Include files directly
require_once 'gusto_api.php';
$config = require __DIR__ . '/gusto_config.php';

// Test data
$week = date('Y-m-d', strtotime('last monday'));
$pay_period_start = $week;
$pay_period_end = date('Y-m-d', strtotime($week . ' +6 days'));
$pay_date = date('Y-m-d', strtotime($week . ' +11 days'));

$pay_payload = [
    'start_date' => $pay_period_start,
    'end_date'   => $pay_period_end,
    'pay_date'   => $pay_date,
    'off_cycle'  => true,
    'off_cycle_reason' => 'Bonus'
];

echo "<h4>1. Testing Company Access First</h4>";
$company_test = gusto_request('GET', "companies/{$config->COMPANY_UUID}");
echo "Company Access - HTTP Code: " . $company_test['http_code'] . "<br>";
if ($company_test['http_code'] === 200) {
    echo "✅ Company access successful<br>";
    echo "Company Name: " . ($company_test['json']['name'] ?? 'Unknown') . "<br>";
} else {
    echo "❌ Company access failed<br>";
    if (isset($company_test['json']['errors'])) {
        foreach ($company_test['json']['errors'] as $error) {
            echo "Error: " . $error['message'] . "<br>";
        }
    }
}

echo "<hr><h4>2. Payload being sent to Gusto:</h4>";
echo "<pre>" . json_encode($pay_payload, JSON_PRETTY_PRINT) . "</pre>";

echo "<h4>3. Making API call to create payroll...</h4>";
$create = gusto_request('POST', "companies/{$config->COMPANY_UUID}/payrolls", $pay_payload);

echo "<h4>4. API Response Details:</h4>";
echo "HTTP Status Code: " . $create['http_code'] . "<br>";
echo "Raw Response: <pre>" . htmlspecialchars($create['raw'] ?? 'No response') . "</pre>";

echo "<h4>5. Analysis:</h4>";
if ($create['http_code'] >= 200 && $create['http_code'] < 300) {
    echo "<div style='color:green;'>✅ API call successful (HTTP {$create['http_code']})</div>";
    
    if (!empty($create['json']['uuid'])) {
        echo "<div style='color:green;'>✅ Payroll UUID received: " . $create['json']['uuid'] . "</div>";
        
        // Test if we can retrieve the payroll immediately
        echo "<h4>6. Testing immediate payroll retrieval...</h4>";
        $retrieve = gusto_request('GET', "payrolls/{$create['json']['uuid']}");
        echo "Retrieval HTTP Code: " . $retrieve['http_code'] . "<br>";
        
        if ($retrieve['http_code'] === 200) {
            echo "<div style='color:green;'>✅ Payroll retrieval successful immediately!</div>";
        } else {
            echo "<div style='color:orange;'>⚠️ Payroll not immediately available (HTTP {$retrieve['http_code']})</div>";
            echo "This is normal - it can take a few minutes for payrolls to process.<br>";
        }
        
    } else {
        echo "<div style='color:red;'>❌ No UUID in response - payroll creation may have failed</div>";
        if (isset($create['json'])) {
            echo "Response JSON: <pre>" . json_encode($create['json'], JSON_PRETTY_PRINT) . "</pre>";
        }
    }
} else {
    echo "<div style='color:red;'>❌ API call failed (HTTP {$create['http_code']})</div>";
    
    if (isset($create['json']['errors'])) {
        echo "<h4>Gusto API Errors:</h4>";
        foreach ($create['json']['errors'] as $error) {
            echo "<div style='color:red;'>- {$error['message']}";
            if (isset($error['error_key'])) {
                echo " (Field: {$error['error_key']})";
            }
            echo "</div>";
        }
    } elseif (isset($create['error'])) {
        echo "<div style='color:red;'>Error: " . $create['error'] . "</div>";
    }
    
    // Show full response for debugging
    echo "<h4>Full Response for Debugging:</h4>";
    echo "<pre>" . htmlspecialchars(print_r($create, true)) . "</pre>";
}

// Test if there are any employees in the company
echo "<hr><h4>7. Testing Company Employees</h4>";
$employees = gusto_request('GET', "companies/{$config->COMPANY_UUID}/employees");
echo "Employees API - HTTP Code: " . $employees['http_code'] . "<br>";
if ($employees['http_code'] === 200 && !empty($employees['json']['employees'])) {
    echo "✅ Found " . count($employees['json']['employees']) . " employees in Gusto<br>";
    foreach ($employees['json']['employees'] as $employee) {
        echo "- " . $employee['first_name'] . " " . $employee['last_name'] . " (" . $employee['uuid'] . ")<br>";
    }
} else {
    echo "❌ No employees found or access denied<br>";
}
?>