<?php
// gusto_api.php
// Requires: db.php (must define $con = mysqli connection) and gusto_config.php
//error_reporting(1);
if (! file_exists(__DIR__.'/gusto_config.php')) {
    die("Missing gusto_config.php - create it from instructions.");
}
$config = require __DIR__ . '/gusto_config.php';

// get latest token row
function _gusto_get_token_row() {
    global $con;
    $q = "SELECT * FROM gusto_tokens ORDER BY id DESC LIMIT 1";
    $res = mysqli_query($con, $q);
    if ($res && $row = mysqli_fetch_assoc($res)) return $row;
    return null;
}

// save token row - FIXED: Uncommented execute statement
function _gusto_save_tokens($access, $refresh, $expires_in) {
    global $con, $config;
    $expires_at = date('Y-m-d H:i:s', time() + intval($expires_in) - intval($config->REFRESH_MARGIN));
    $stmt = mysqli_prepare($con, "INSERT INTO gusto_tokens (access_token, refresh_token, expires_at) VALUES (?, ?, ?)");
    mysqli_stmt_bind_param($stmt, "sss", $access, $refresh, $expires_at);
    mysqli_stmt_execute($stmt); // FIXED: This was commented out
    mysqli_stmt_close($stmt);
    
    // Log token refresh
    error_log("Gusto tokens refreshed - Expires: " . $expires_at);
}

// Enhanced token refresh with better error handling
function _gusto_refresh_with_refresh_token($refresh_token) {
    global $config;
    $url = $config->OAUTH_TOKEN_URL;
    $post = http_build_query([
        'grant_type'    => 'refresh_token',
        'refresh_token' => $refresh_token,
        'client_id'     => $config->CLIENT_ID,
        'client_secret' => $config->CLIENT_SECRET
    ]);

    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => $post,
        CURLOPT_HTTPHEADER => ['Content-Type: application/x-www-form-urlencoded'],
        CURLOPT_TIMEOUT => 30,
        CURLOPT_SSL_VERIFYPEER => true
    ]);
    
    $raw = curl_exec($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        error_log("Gusto token refresh curl error: " . $error);
        return false;
    }
    
    $json = @json_decode($raw, true);
    if ($code >= 200 && $code < 300 && isset($json['access_token'])) {
        _gusto_save_tokens($json['access_token'], $json['refresh_token'] ?? $refresh_token, $json['expires_in'] ?? 3600);
        error_log("Gusto token refresh successful");
        return $json['access_token'];
    } else {
        error_log("Gusto token refresh failed. HTTP $code: " . $raw);
        return false;
    }
}

// Enhanced token validation with proactive refresh
function get_gusto_access_token() {
    global $config;
    
    // Load latest token
    $row = _gusto_get_token_row();
    if (!$row) {
        error_log("No Gusto token found in database");
        return null;
    }
    
    // Check if token is expired or about to expire
    $expires_time = strtotime($row['expires_at']);
    $current_time = time();
    $time_until_expiry = $expires_time - $current_time;
    
    // If token is still valid for more than 5 minutes, return it
    if (!empty($row['access_token']) && $time_until_expiry > 300) {
        return $row['access_token'];
    }
    
    // Token is expired or about to expire, try refresh
    if (!empty($row['refresh_token'])) {
        error_log("Gusto token expiring in {$time_until_expiry}s, attempting refresh...");
        $new_token = _gusto_refresh_with_refresh_token($row['refresh_token']);
        if ($new_token) {
            return $new_token;
        }
    }
    
    // No valid token available
    error_log("No valid Gusto access token available");
    return null;
}

// generic request helper that auto-refreshes once on 401
function gusto_request($method, $path, $body = null, $extra_headers = []) {
    global $config;
    $access = get_gusto_access_token();
    if (!$access) return ['http_code' => 401, 'raw' => 'No access token (authorize first)', 'json'=>null];

    $url = rtrim($config->GUSTO_API_BASE, '/') . '/' . ltrim($path, '/');
    $ch = curl_init($url);
    $method = strtoupper($method);

    $headers = array_merge([
        "Authorization: Bearer {$access}"
    ], $extra_headers);
    if ($body !== null && !in_array('Content-Type: application/json', $headers)) {
        $headers[] = "Content-Type: application/json";
    }

    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_CUSTOMREQUEST => $method,
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_TIMEOUT => 60,
        CURLOPT_SSL_VERIFYPEER => true
    ]);
    
    if ($body !== null) {
        curl_setopt($ch, CURLOPT_POSTFIELDS, is_string($body) ? $body : json_encode($body));
    }
    
    $raw = curl_exec($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    $json = @json_decode($raw, true);

    // If 401, try refreshing once and retry
    if ($code == 401) {
        error_log("API request returned 401, attempting token refresh and retry...");
        // attempt refresh
        $row = _gusto_get_token_row();
        if ($row && !empty($row['refresh_token'])) {
            $refreshed = _gusto_refresh_with_refresh_token($row['refresh_token']);
            if ($refreshed) {
                // retry
                $access = $refreshed;
                $ch = curl_init($url);
                $headers = array_merge(["Authorization: Bearer {$access}"], $extra_headers);
                if ($body !== null && !in_array('Content-Type: application/json', $headers)) {
                    $headers[] = "Content-Type: application/json";
                }
                
                curl_setopt_array($ch, [
                    CURLOPT_RETURNTRANSFER => true,
                    CURLOPT_CUSTOMREQUEST => $method,
                    CURLOPT_HTTPHEADER => $headers,
                    CURLOPT_TIMEOUT => 60,
                    CURLOPT_SSL_VERIFYPEER => true
                ]);
                
                if ($body !== null) {
                    curl_setopt($ch, CURLOPT_POSTFIELDS, is_string($body) ? $body : json_encode($body));
                }
                
                $raw = curl_exec($ch);
                $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                curl_close($ch);
                $json = @json_decode($raw, true);
                
                error_log("Retry after refresh returned HTTP: $code");
            }
        }
    }

    return ['http_code' => $code, 'raw' => $raw, 'json' => $json];
}

// NEW: Submit to Gusto function for shared use
function submitToGusto($con, $config, $week) {
    log_message("Submitting payroll to Gusto for week: $week");
    
    // Fetch rows for that week
    $q = "SELECT wr.*, u.gusto_employee_uuid, u.vName as employee_name 
          FROM weekly_reports wr 
          JOIN usermst u ON u.id = wr.employee_id 
          WHERE wr.week_start = '" . mysqli_real_escape_string($con, $week) . "' AND wr.gusto_submitted = 0 AND u.is_active = 1";
    
    $res = mysqli_query($con, $q);
    $rows = [];
    while ($r = mysqli_fetch_assoc($res)) $rows[] = $r;
    
    if (count($rows) == 0) { 
        log_message("No unsubmitted reports for $week");
        return "No unsubmitted reports for $week";
    }

    $pay_period_start = $week;
    $pay_period_end = date('Y-m-d', strtotime($week . ' +6 days'));
    $pay_date = date('Y-m-d', strtotime($week . ' +11 days'));
    
    // Build employee compensations array for the payroll
    $employee_compensations = [];
    $compensation_errors = [];
    
    foreach ($rows as $row) {
        if (empty($row['gusto_employee_uuid'])) {
            $compensation_errors[] = "Missing gusto_employee_uuid for employee ID: " . $row['employee_id'];
            continue;
        }
        
        // Calculate earnings
        $regular_hours = $row['hours_total'] - $row['solo_hours'];
        $regular_earnings = $regular_hours * ($row['hourly_rate'] ?: 15.00);
        $solo_bonus_earnings = $row['solo_hours'] * ($row['solo_bonus_rate'] ?: 3.00);
        $mileage_reimbursement = $row['miles'] * ($row['mileage_rate'] ?: 0.655);
        $miscellaneous_pay = $row['miscellaneous_pay'] ?? 0; // NEW: Include miscellaneous pay
        
        $earnings = [];
        
        // Add regular hours earning
        if ($regular_hours > 0) {
            $earnings[] = [
                'hours' => (string)number_format($regular_hours, 2),
                'rate' => (string)number_format($row['hourly_rate'] ?: 15.00, 2),
                'earning_type' => 'Regular',
                'amount' => (string)number_format($regular_earnings, 2)
            ];
        }
        
        // Add solo bonus earning
        if ($row['solo_hours'] > 0) {
            $earnings[] = [
                'hours' => (string)number_format($row['solo_hours'], 2),
                'rate' => (string)number_format($row['solo_bonus_rate'] ?: 3.00, 2),
                'earning_type' => 'Bonus',
                'amount' => (string)number_format($solo_bonus_earnings, 2)
            ];
        }
        
        // Add mileage reimbursement
        if ($row['miles'] > 0) {
            $earnings[] = [
                'amount' => (string)number_format($mileage_reimbursement, 2),
                'earning_type' => 'Reimbursement'
            ];
        }
        
        // NEW: Add miscellaneous as a separate bonus if amount > 0
        if ($miscellaneous_pay > 0) {
            $earnings[] = [
                'amount' => (string)number_format($miscellaneous_pay, 2),
                'earning_type' => 'Bonus',
                'description' => 'Miscellaneous'
            ];
        }
        
        $employee_compensations[] = [
            'employee_uuid' => $row['gusto_employee_uuid'],
            'earnings' => $earnings
        ];
    }
    
    if (empty($employee_compensations)) {
        log_message("No valid employee compensations to submit");
        return "No valid employee compensations to submit";
    }
    
    // Create payroll WITH employee compensations
    $pay_payload = [
        'start_date' => $pay_period_start,
        'end_date'   => $pay_period_end,
        'pay_date'   => $pay_date,
        'off_cycle'  => true,
        'off_cycle_reason' => 'Bonus',
        'employee_compensations' => $employee_compensations
    ];
    
    log_message("Creating payroll in Gusto...");
    $create = gusto_request('POST', "companies/{$config->COMPANY_UUID}/payrolls", $pay_payload);
    
    $api_response_json = json_encode($create);
    
    if ($create['http_code'] >= 200 && $create['http_code'] < 300 && !empty($create['json']['uuid'])) {
        $payroll_uuid = $create['json']['uuid'];
        
        // Store payroll with API response
        $stmt = mysqli_prepare($con, "INSERT INTO payrolls (payroll_uuid, company_uuid, week_start, pay_period_start, pay_period_end, pay_date, api_response, status) VALUES (?, ?, ?, ?, ?, ?, ?, 'created')");
        mysqli_stmt_bind_param($stmt, "sssssss", $payroll_uuid, $config->COMPANY_UUID, $week, $pay_period_start, $pay_period_end, $pay_date, $api_response_json);
        mysqli_stmt_execute($stmt);
        mysqli_stmt_close($stmt);

        // Mark all reports as submitted
        foreach ($rows as $row) {
            $update_stmt = mysqli_prepare($con, "UPDATE weekly_reports SET gusto_submitted=1, gusto_payroll_uuid=? WHERE id=?");
            mysqli_stmt_bind_param($update_stmt, "si", $payroll_uuid, $row['id']);
            mysqli_stmt_execute($update_stmt);
            mysqli_stmt_close($update_stmt);
        }

        $result = "✅ Payroll created successfully! UUID: " . substr($payroll_uuid, 0, 8) . "...";
        log_message($result);
        
        // Try to prepare the payroll
        $prepare = gusto_request('PUT', "companies/{$config->COMPANY_UUID}/payrolls/{$payroll_uuid}/prepare");
        if ($prepare['http_code'] >= 200 && $prepare['http_code'] < 300) {
            $result .= " Payroll prepared successfully.";
            mysqli_query($con, "UPDATE payrolls SET status='prepared' WHERE payroll_uuid='" . mysqli_real_escape_string($con, $payroll_uuid) . "'");
            log_message("Payroll prepared successfully");
        } else {
            log_message("Payroll preparation may need manual intervention");
        }
        
        // Show any warnings about missing employee mappings
        if (!empty($compensation_errors)) {
            $result .= " Warnings: " . implode("; ", $compensation_errors);
        }
        
        return $result;
        
    } else {
        // Store the failed attempt for debugging
        $stmt = mysqli_prepare($con, "INSERT INTO payrolls (payroll_uuid, company_uuid, week_start, pay_period_start, pay_period_end, pay_date, api_response, status) VALUES (?, ?, ?, ?, ?, ?, ?, 'failed')");
        $failed_uuid = 'failed-' . time();
        mysqli_stmt_bind_param($stmt, "sssssss", $failed_uuid, $config->COMPANY_UUID, $week, $pay_period_start, $pay_period_end, $pay_date, $api_response_json);
        mysqli_stmt_execute($stmt);
        mysqli_stmt_close($stmt);
        
        $error_msg = "❌ Failed to create payroll (HTTP {$create['http_code']}). ";
        if (isset($create['json']['errors'])) {
            foreach ($create['json']['errors'] as $error) {
                $error_msg .= "{$error['message']}. ";
            }
        } else {
            $error_msg .= "Check the debug information for details.";
        }
        log_message($error_msg);
        return $error_msg;
    }
}

// NEW: Logging function for shared use
function log_message($message) {
    $timestamp = date('Y-m-d H:i:s');
    $log_entry = "[$timestamp] $message\n";
   // echo $log_entry;
    file_put_contents(__DIR__ . '/payroll_automation.log', $log_entry, FILE_APPEND);
}

// gusto_api.php with enhanced custom fields support
// Add this function after existing code

function submitToGustoEnhanced($con, $config, $weekly_report_main_id) {
    log_message("Submitting payroll to Gusto for week: $week (Enhanced with Custom Fields)");
    
    // Fetch rows for that week with custom fields
    $q = "SELECT wr.*, u.gusto_employee_uuid, u.vName as employee_name,
                 (SELECT SUM(calculated_pay) 
                  FROM weekly_report_custom_fields wrcf 
                  WHERE wrcf.weekly_report_id = wr.id) as custom_fields_total
          FROM weekly_reports wr 
          JOIN usermst u ON u.id = wr.employee_id
          WHERE wr.weekly_report_main_id = '" . mysqli_real_escape_string($con, $weekly_report_main_id) . "' 
          AND wr.gusto_submitted = 0 
          -- AND u.is_active = 1";
    
    $res = mysqli_query($con, $q);
    $rows = [];
    while ($r = mysqli_fetch_assoc($res)) $rows[] = $r;
    $sql2 = "SELECT * FROM payrolls
          WHERE weekly_report_main_id = '" . mysqli_real_escape_string($con, $weekly_report_main_id) . "'";
    
    $res2 = mysqli_query($con, $sql2);
    $r2 = mysqli_num_rows($res2);
    if(!$r2) {
        $sql3 = "SELECT * FROM weekly_reports_main
              WHERE id = '" . mysqli_real_escape_string($con, $weekly_report_main_id) . "'";
        
        $res3 = mysqli_query($con, $sql3);
        $r3 = mysqli_fetch_assoc($res3);
        $pay_period_start = $r3["week_start"];
        $pay_period_end = $r3["week_end"];
        $pay_date = date('Y-m-d', strtotime($pay_period_start . ' +11 days'));
        
        if (count($rows) == 0) { 
            log_message("No unsubmitted reports for $week");
            return "No unsubmitted reports for $week";
        }
    
        //$pay_period_start = $week;
        //$pay_period_end = date('Y-m-d', strtotime($week . ' +6 days'));
        //$pay_date = date('Y-m-d', strtotime($week . ' +11 days'));
        
        // Build employee compensations array
        $employee_compensations = [];
        $compensation_errors = [];
        
        foreach ($rows as $row) {
            if (empty($row['gusto_employee_uuid'])) {
                //$compensation_errors[] = "Missing gusto_employee_uuid for employee ID: " . $row['employee_id'];
                //continue;
                $payload = [
                    "first_name" => "John",
                    "last_name"  => "Doe",
                    "email"      => "john.doe@example.com",
                    "ssn"        => "123-45-6789",
                    "date_of_birth" => "1990-05-12",
                    "home_address" => [
                        "street_1" => "123 Main St",
                        "street_2" => "",
                        "city"     => "Los Angeles",
                        "state"    => "CA",
                        "zip"      => "90001",
                        "country"  => "USA"
                    ]
                ];
                exit;
                $create = gusto_request('POST', "companies/{$config->COMPANY_UUID}/employees", $pay_payload);
            }
            
            // Calculate standard earnings
            $regular_hours = $row['hours_total'];
            $regular_earnings = $regular_hours * ($row['hourly_rate'] ?: 16.00);
            $solo_bonus_earnings = $row['solo_hours'] * ($row['solo_bonus_rate'] ?: 3.00);
            $mileage_reimbursement = $row['miles'] * ($row['mileage_rate'] ?: 0.25);
            $miscellaneous_pay = $row['miscellaneous_pay'] ?? 0;
            $custom_fields_total = $row['custom_fields_total'] ?? 0;
            
            $earnings = [];
            
            // Add regular hours earning
            if ($regular_hours > 0) {
                $earnings[] = [
                    'hours' => (string)number_format($regular_hours, 2),
                    'rate' => (string)number_format($row['hourly_rate'] ?: 16.00, 2),
                    'earning_type' => 'Regular',
                    'amount' => (string)number_format($regular_earnings, 2)
                ];
            }
            
            // Add solo bonus earning
            if ($row['solo_hours'] > 0) {
                $earnings[] = [
                    'hours' => (string)number_format($row['solo_hours'], 2),
                    'rate' => (string)number_format($row['solo_bonus_rate'] ?: 3.00, 2),
                    'earning_type' => 'Bonus',
                    'amount' => (string)number_format($solo_bonus_earnings, 2)
                ];
            }
            
            // Add mileage reimbursement
            if ($row['miles'] > 0) {
                $earnings[] = [
                    'amount' => (string)number_format($mileage_reimbursement, 2),
                    'earning_type' => 'Reimbursement',
                    'description' => 'Mileage reimbursement'
                ];
            }
            
            // Add miscellaneous as bonus
            if ($miscellaneous_pay > 0) {
                $earnings[] = [
                    'amount' => (string)number_format($miscellaneous_pay, 2),
                    'earning_type' => 'Bonus',
                    'description' => 'Miscellaneous'
                ];
            }
            
            // Add custom fields as separate earnings
            if ($custom_fields_total > 0) {
                // Get custom field details for this report
                $cf_query = mysqli_query($con, "
                    SELECT cpf.field_name, wrf.calculated_pay
                    FROM weekly_report_custom_fields wrf
                    JOIN custom_payroll_fields cpf ON cpf.id = wrf.custom_field_id
                    WHERE wrf.weekly_report_id = {$row['id']}
                    AND wrf.calculated_pay > 0
                ");
                
                while ($cf = mysqli_fetch_assoc($cf_query)) {
                    $earnings[] = [
                        'amount' => (string)number_format($cf['calculated_pay'], 2),
                        'earning_type' => 'Bonus',
                        'description' => $cf['field_name']
                    ];
                }
            }
    
            // if(!$row['gusto_employee_uuid']) {
            //     print_r($row);
            // }
            $employee_compensations[] = [
                'employee_uuid' => $row['gusto_employee_uuid'],
                'earnings' => $earnings
            ];
            
            
            // Log custom fields details
            if ($custom_fields_total > 0) {
                log_message("Employee {$row['employee_name']}: Custom fields total: $" . number_format($custom_fields_total, 2));
            }
        }
        
        if (empty($employee_compensations)) {
            log_message("No valid employee compensations to submit");
            return "No valid employee compensations to submit";
        }
        
        // Create payroll WITH employee compensations
        $pay_payload = [
            'start_date' => $pay_period_start,
            'end_date'   => $pay_period_end,
            'pay_date'   => $pay_date,
            'off_cycle'  => true,
            'off_cycle_reason' => 'Bonus',
            'employee_compensations' => $employee_compensations
        ];
        echo "<pre>";
        print_r($pay_payload);
        echo "</pre>";
        exit;
        log_message("Creating enhanced payroll in Gusto...");
        $create = gusto_request('POST', "companies/{$config->COMPANY_UUID}/payrolls", $pay_payload);
        
        $api_response_json = json_encode($create);
        
        if ($create['http_code'] >= 200 && $create['http_code'] < 300 && !empty($create['json']['uuid'])) {
            $payroll_uuid = $create['json']['uuid'];
            
            // Store payroll with API response
            $stmt = mysqli_prepare($con, "INSERT INTO payrolls (payroll_uuid, company_uuid, weekly_report_main_id, pay_date, api_response, status) VALUES (?, ?, ?, ?, ?, 'created')");
            mysqli_stmt_bind_param($stmt, "sssss", $payroll_uuid, $config->COMPANY_UUID, $row['weekly_report_main_id'], $pay_date, $api_response_json);
            mysqli_stmt_execute($stmt);
            mysqli_stmt_close($stmt);
    
            // Mark all reports as submitted
            foreach ($rows as $row) {
                $update_stmt = mysqli_prepare($con, "UPDATE weekly_reports SET gusto_submitted=1, gusto_payroll_uuid=? WHERE id=?");
                mysqli_stmt_bind_param($update_stmt, "si", $payroll_uuid, $row['id']);
                mysqli_stmt_execute($update_stmt);
                mysqli_stmt_close($update_stmt);
                
                // Log custom fields submission
                if ($row['custom_fields_total'] > 0) {
                    log_message("✓ Report ID {$row['id']} submitted with custom fields: $" . number_format($row['custom_fields_total'], 2));
                }
            }
    
            $result = "Payroll created successfully with custom fields! UUID: " . substr($payroll_uuid, 0, 8) . "...";
            log_message($result);
            
            // Try to prepare the payroll
            $prepare = gusto_request('PUT', "companies/{$config->COMPANY_UUID}/payrolls/{$payroll_uuid}/prepare");
            if ($prepare['http_code'] >= 200 && $prepare['http_code'] < 300) {
                $result .= " Payroll prepared successfully.";
                mysqli_query($con, "UPDATE payrolls SET status='prepared' WHERE payroll_uuid='" . mysqli_real_escape_string($con, $payroll_uuid) . "'");
                mysqli_query($con, "UPDATE weekly_reports_main SET submitted_to_gutso='1' WHERE id='" . mysqli_real_escape_string($con, $$weekly_report_main_id) . "'");
                log_message("Payroll prepared successfully");
            }
            
            // Show any warnings about missing employee mappings
            if (!empty($compensation_errors)) {
                $result .= " Warnings: " . implode("; ", $compensation_errors);
            }
            
            return $result;
            
        } else {
            // Store the failed attempt for debugging
            $stmt = mysqli_prepare($con, "INSERT INTO payrolls (payroll_uuid, company_uuid, weekly_report_main_id, pay_date, api_response, status) VALUES (?, ?, ?, ?, ?, 'failed')");
            $failed_uuid = 'failed-' . time();
            mysqli_stmt_bind_param($stmt, "sssss", $failed_uuid, $config->COMPANY_UUID, $row["weekly_report_main_id"], $pay_date, $api_response_json);
            mysqli_stmt_execute($stmt);
            mysqli_stmt_close($stmt);
            
            $error_msg = "Failed to create payroll (HTTP {$create['http_code']}). ";
            if (isset($create['json']['errors'])) {
                foreach ($create['json']['errors'] as $error) {
                    $error_msg .= "{$error['message']}. ";
                }
            } else {
                $error_msg .= "Check the debug information for details.";
            }
            log_message($error_msg);
            return $error_msg;
        }
    } else {
         mysqli_query($con, "UPDATE weekly_reports_main SET submitted_to_gutso='1' WHERE id='" . mysqli_real_escape_string($con, $weekly_report_main_id) . "'");
        $error_msg .= "Payroll has already run for the week $weekly_report_main_id.";
        log_message($error_msg);
        return $error_msg;
    }
}